use utf8;
use strict;
use warnings FATAL => 'all';

package Xyzzy::CAS::Request;

use Xyzzy::Util qw(uri_escape_auto);
use Xyzzy::CAS::Session;
use LWP;
use URI;

use Xyzzy::Request -self;

field crypto => sub { shift->cfg->crypto };

param ticket => sub { undef $_ unless defined && /^ST-[a-z0-9-]+$/i };

sub build_url() {
	my $url = shift;
	$url =~ s/(\#.*)//;
	my $frag = $1 // '';
	my $sep = index($url, '?') == -1 ? '?' : '&';
	my @ret = ($url);
	my $q = $_[0];
	@_ = %$q if ref $q;
	while(@_) {
		my $key = shift;
		my $val = shift;
		push @ret, $sep, uri_escape_auto($key), '=', uri_escape_auto($val);
		$sep = '&';
	}
	return join('', @ret, $frag);
}

sub fetch_url {
	my $url = build_url(@_);
	our $ua //= new LWP::UserAgent;
	my $res = $ua->get($url);
	die "got ".$res->status_line." while fetching '$url'\n" unless $res->is_success;
	return $res->decoded_content;
}

field cas_clean_url => sub {
	my $self = shift;
	my $cfg = $self->cfg;

	my $url = URI->new($self->self_url)->canonical;
	$url->userinfo(undef);

	my $has_port = $url->authority =~ /:\d*$/;
	my $is_localhost = $url->host =~ /^(?:localhost|127\.0\.0\.1)$/;
	my $is_hostname = $url->host !~ /\./;

	if(my $domain = $cfg->domain) {
		$url->host($domain)
			if !$is_localhost && ($is_hostname || $cfg->force_domain);
	}
	
	$url->scheme('https')
		if $url->scheme eq 'http' && $cfg->force_https && !$is_localhost && !$has_port;

	# remove ticket=... from the query part
	my $query = $url->query;
	if(defined $query) {
		$query = join('&', grep { !/^(?:$|ticket(?:=|$))/ } split(/[&;]/, $query));
		undef $query if $query eq '';
		$url->query($query);
	}

	return $url->canonical->as_string;
};

sub create_session {
	my $noise = 's'.$self->remote_addr;
	my ($stamp, undef, $token) = $self->crypto->create_token($noise, @_);
	return new Xyzzy::CAS::Session(token => $token, timestamp => $stamp, uid => shift);
}

sub session {
	return $self->new_session->uid;
}

field new_session => sub {
	my $self = shift;
	my $uid = $self->valid_ticket;
	return $self->create_session($uid)
		if defined $uid;
	my $cur = $self->cur_session;
	return $cur if $cur->token;
	return $self->create_session;
};

field cur_session => sub {
	my $self = shift;
	my $token = $self->cookie('session');
	if(defined $token) {
		my $noise = 's'.$self->remote_addr;
		my $timeout = $self->cfg->session_timeout;
		if(my ($stamp, undef, $uid) = eval { $self->crypto->check_token($noise, $token, $timeout) }) {
			return new Xyzzy::CAS::Session(token => $token, timestamp => $stamp, uid => $uid);
		}
		warn $@;
	}
	return new Xyzzy::CAS::Session(token => undef, timestamp => undef, uid => undef);
};

sub cas_redirect {
	my $url = shift;
	my $status = new Xyzzy::Status(302);
	my $res = $status->response;
	$res->addheader(Location => $url);
	die $res;
}

sub valid_ticket {
	my $ticket = $self->ticket;
	return undef unless defined $ticket;
	my $url = $self->cas_clean_url;
	my $res = $self->fetch_url($self->cfg->validate_url, ticket => $ticket, service => $url);
	return undef unless $res =~ /^yes\n([A-Za-z0-9_-]+)\n$/;
	return $1;
}

sub login {
	my $uid = $self->session // $self->valid_ticket;
	return $uid if defined $uid;
	my $url = $self->cas_clean_url;
	$self->cas_redirect(build_url($self->cfg->login_url, service => $url));
}

param nonce => sub {
	my $self = shift;
	my $session = $self->cur_session->token;
	$_ = eval { $self->crypto->check_token('n'.$session, $_, $self->cfg->request_timeout); 1 };
};

sub create_nonce {
	my $session = $self->new_session->token;
	return scalar $self->crypto->create_token('n'.$session);
}
