use utf8;
use strict;
use warnings FATAL => 'all';

package Xyzzy::LDAP;

use URI;
use Net::LDAP;
use Net::LDAP::Filter;

use Clarity -self;

field cfg;
field server => sub { shift->cfg->ldap_server };
field base => sub { shift->cfg->ldap_base };
field filter => sub { shift->cfg->ldap_filter };
field attribute => sub { shift->cfg->ldap_attribute };
field username => sub { shift->cfg->ldap_username };
field password => sub { shift->cfg->ldap_password };
field secure => sub { shift->cfg->ldap_secure };
field cafile => sub { shift->cfg->ldap_cafile };
field capath => sub { shift->cfg->ldap_capath };

sub bind {
	my $ldap = shift // $self->connection;
	my $username = $self->username;
	return $ldap->bind($username, password => $self->password)
		if $username;
	return $ldap->bind;
}

sub on_error {
	my $obj = shift;
	my $error = $obj->error;
	my $code = $obj->code;
	die $error if ref $error;
	if($error =~ /^.* at \S+ line \d+\.?\n\z/) {
		# evil: mess with Carp internals (if they exist) to get a clean stack trace
		if($Carp::{caller_info}) {
			my $level = 2;
			while(my %info = Carp::caller_info(++$level)) {
				$error .= "\t$info{sub_name} called at $info{file} line $info{line}\n";
			}
			die $error;
		} else {
			local $Carp::CarpLevel = 2;
			confess($error);
		}
	}
	die "$error ($code)\n" if defined $code && $code > 0;
	die "$error\n";
}

sub connection {
	# create a, possible cached, LDAP client object
	my $ldap = $self->{connection};
	if(defined $ldap) {
		eval { $self->bind($ldap) };
		if($@) {
			my $server = $self->server;
			warn "LDAP server $server: $@"
				unless $@ =~ /^Unexpected EOF /;
			undef $ldap;
			delete $self->{connection};
		}
	}
	unless(defined $ldap) {
		my $server = $self->server;
		my %options = (onerror => sub { $self->on_error(@_) }, timeout => 10, inet6 => 1);
		my $host = $server;
		if(index($server, '://') == -1) {
			$host =~ s/:.*//;
		} else {
			my $uri = new URI($server);
			$host = $uri->host;
		}
		if($host eq 'localhost' || !$self->secure) {
			$ldap = new Net::LDAP($server, %options)
				or die "Connecting to $server: $@";
		} else {
			my %ssl = (
					verify => 'require',
					capath => $self->capath,
					cafile => $self->cafile,
				);
			warn "neither LDAPCAfile nor LDAPCApath configured\n"
				unless $ssl{cafile} || $ssl{capath};
			$ldap = new Net::LDAP($server, %options, %ssl)
				or die "Connecting to $server: $@";
			$ldap->start_tls(%ssl)
				unless $ldap->cipher;
			die "STARTTLS failed on LDAP server $server\n"
				unless $ldap->cipher;
			die "Can't verify LDAP server name as '$host'\n"
				unless $ldap->socket->verify_hostname($host, 'ldap');
		}
		$self->bind($ldap);
		$self->{connection} = $ldap;
	}

	return $ldap;
}

sub search {
	my ($uid, $attrs) = @_;

	return undef unless defined $uid;
	$attrs = [$attrs] unless ref $attrs;

	# attribute to search on
	my $attribute = $self->attribute;
	my $struct = {equalityMatch => {attributeDesc => $attribute, assertionValue => $uid}};

	# create a, possible cached, filter
	my $filter = $self->filter;
	$struct = {and => [$struct, $filter]}
		if $filter;

	my $search = bless($struct, 'Net::LDAP::Filter');

	return $self->connection->search(
			base => $self->base,
			filter => $search,
			attrs => $attrs
		);
}

sub authenticate {
	my ($uid, $passwd) = @_;

	my $ldap = $self->connection;

	my $attribute = $self->attribute;
	my $res = $self->search($uid, [$attribute]);

	unless($res->count) {
		warn "Invalid username\n";
		return undef;
	}

	undef $uid;
	foreach my $entry ($res->entries) {
		my $dn = $entry->dn;
		eval {
			$ldap->bind($dn, password => $passwd);
			$self->bind($ldap);
			$uid = $entry->get_value($attribute);
		};
		last if defined $uid;
		warn "$dn: $@";
	}

	return $uid;
}

const schema => sub { shift->connection->schema };
