use utf8;
use strict;
use warnings FATAL => 'all';

package Xyzzy::Status;

use Clarity -self;

use Xyzzy::Response;
use Xyzzy::Util qw(htmlentities);

field code;
field message;

my %statuscodes = (
	100 => 'Continue',
	101 => 'Switching protocols',
	200 => 'OK',
	102 => 'Processing',
	201 => 'Created',
	202 => 'Accepted',
	203 => 'Non-authoritative information',
	204 => 'No content',
	205 => 'Reset content',
	206 => 'Partial content',
	207 => 'Multi-status',
	300 => 'Multiple choices',
	301 => 'Moved permanently',
	302 => 'Moved temporarily',
	303 => 'See other',
	304 => 'Not modified',
	305 => 'Use proxy',
	306 => 'Switch proxy',
	307 => 'Temporary redirect',
	400 => 'Bad request',
	401 => 'Authentication required',
	402 => 'Payment required',
	403 => 'Forbidden',
	404 => 'Page not found',
	405 => 'Method not allowed',
	406 => 'Not acceptable',
	407 => 'Proxy authentication required',
	408 => 'Request timeout',
	409 => 'Conflict',
	410 => 'Gone',
	411 => 'Length required',
	412 => 'Precondition failed',
	413 => 'Request entity too large',
	414 => 'Request URI too long',
	415 => 'Unsupported media type',
	416 => 'Requested range not satisfiable',
	417 => 'Expectation failed',
	418 => "I'm a teapot",
	422 => 'Unprocessable entity',
	423 => 'Locked',
	424 => 'Failed dependency',
	425 => 'Unordered collection',
	426 => 'Upgrade required',
	449 => 'Retry with',
	450 => 'Blocked by parental controls',
	451 => 'Unavailable for legal reasons',
	500 => 'Internal server error',
	501 => 'Not implemented',
	502 => 'Bad gateway',
	503 => 'Service unavailable',
	504 => 'Gateway timeout',
	505 => 'HTTP version not supported',
	506 => 'Variant also negotiates',
	507 => 'Insufficient storage',
	509 => 'Bandwidth limit exceeded',
	510 => 'Not extended'
);

my %statusclasses = (
	1 => 'Informational',
	2 => 'Successful',
	3 => 'Redirection',
	4 => 'Client error',
	5 => 'Server error'
);

sub html {
	my $message = htmlentities($self->toString);
	return "<!DOCTYPE html>\n<html><head>\n<title>$message</title>\n</head><body>\n<h1>$message</h1>\n</body></html>\n";
}

sub response {
	return new Xyzzy::Response(status => $self, content => $self->html);
}

sub toString {
	my $code = $self->code;
	my $message = $self->message;
	return "$code $message"
		if defined $message && $message ne '';
	return "$code";
}

sub ok {
	return $self->code < 400
}

sub new {
	my ($code, $message) = @_;
	$code //= 200;
	$code =~ /^[1-5]\d\d$/
		or confess("Not a valid HTTP status code: $code");
	$code = int($code);
	if(defined $message) {
		$message =~ s/^\s+|\s+$//g;
		$message =~ s/\s+/ /g;
	} else {
		$message = $statuscodes{$code};
		$message = $statusclasses{int($code/100)}
			unless defined $message;
	}
	return super(code => $code, message => $message);
}

use overload
	'""' => \&toString,
	'0+' => \&code,
	bool => \&ok,
	fallback => 1;
