#! /usr/bin/perl

use strict;
use warnings FATAL => 'all';

package LWP;

use Clarity -base;

package HTTP::Response;

use Clarity -self;

use constant is_success => 1;
field decoded_content;

package LWP::UserAgent;

use Clarity -self;
use Data::Dumper;

sub get {
	my $url = shift;
	$url =~ /ticket=ST-([^&;]*)/;
	my $uid = $1;
	my $res = $uid eq '' ? "no\n\n" : "yes\n$uid\n";
	return new HTTP::Response(decoded_content => $res);
}

package main;

use Test::More;

BEGIN { use_ok('Xyzzy::CAS') or BAIL_OUT('need Xyzzy::CAS to run') }

new_ok('Xyzzy::CAS') or BAIL_OUT('need a Xyzzy::CAS object to run');

package CASResult;

use Xyzzy::Handler -self;

sub handle {
	my $req = shift;
	my $res = new Xyzzy::Response(mimetype => 'text/plain', content => $req->login);
	my @redir = $req->param('redirect');
	if(@redir) {
		$res->status(302);
		$res->setheader(Location => 'http://example.com');
	}
	return $res;
}

package CASTester;

use Xyzzy -self;

use Xyzzy::Crypto::Config -mixin;
use Xyzzy::CAS::Config -mixin;

sub handler {
	my $r = new CASResult(cfg => $self);
	return new Xyzzy::Directory(cfg => $self, subdirs => {
		snoop => new Xyzzy::CAS(cfg => $self, handler => $r, snoop => 1),
		nosnoop => new Xyzzy::CAS(cfg => $self, handler => $r),
	});
}

package main;

use Test::Xyzzy qw(xyzzy_from_hash xyzzy_request xyzzy_request_as_cgi);

my $app = xyzzy_from_hash(
	Application => 'CASTester',
	CryptoSecret => 'secret',
	CASBaseURL => 'http://sso.zo.oi',
);

my $handler = $app->handler;

isa_ok($handler, 'Xyzzy::Handler');

sub parse_cgi {
	my ($headers, $body) = split(/\015\012\015\012/, shift, 2);
	my @headers = split(/\015\012/, $headers);
	return $body, @headers;
}

sub parse_cookies {
	return map { /^Set-Cookie:\s+([^;=]+)=([^;]+)/ ? ($1, $2) : () } @_;
}

sub bake_cookies {
	my %cookies = @_;
	my @cookies;
	while(my ($name, $value) = each %cookies) {
		push @cookies, "$name=$value";
	}
	return join('; ', @cookies);
}

do {
	my $res = xyzzy_request_as_cgi($handler, '/nosnoop');
	my ($body, @headers) = parse_cgi($res);
	ok((grep { m{^Location: http://sso\.zo\.oi/login\?} } @headers), "headers contain location to login url");
	ok((grep { m{^Set-Cookie: session=[^; ]+} } @headers), "headers contain session cookie");
};

do {
	my $res = xyzzy_request_as_cgi($handler, '/nosnoop', 'redirect&ticket=ST-foo');
	my ($body, @headers) = parse_cgi($res);
	ok((grep { $_ eq 'Location: http://localhost/nosnoop?redirect' } @headers), "headers contain location to application base URL without ticket");
	ok((grep { m{^Set-Cookie: session=[^; ]+} } @headers), "headers contain session cookie");
};

do {
	my $res = xyzzy_request_as_cgi($handler, '/snoop', 'redirect&ticket=ST-foo');
	my ($body, @headers) = parse_cgi($res);
	ok((grep { $_ eq 'Location: http://example.com' } @headers), "headers contain location to url as generated by the application");
	ok((grep { m{^Set-Cookie: session=[^; ]+} } @headers), "headers contain session cookie");
};

my %jar;

do {
	my $res = xyzzy_request_as_cgi($handler, '/nosnoop', 'ticket=ST-foo');
	my ($body, @headers) = parse_cgi($res);
	ok((grep { $_ eq 'Location: http://localhost/nosnoop' } @headers), "headers contain location to application base URL without ticket");
	ok((grep { m{^Set-Cookie: session=[^; ]+} } @headers), "headers contain session cookie");
	%jar = parse_cookies(@headers);
};

do {
	my $res = xyzzy_request_as_cgi($handler, '/nosnoop', undef, HTTP_COOKIE => bake_cookies(%jar));
	my ($body, @headers) = parse_cgi($res);
	is($body, 'foo', "result is the uid");
	ok(!(grep { m{^Set-Cookie: session=[^; ]+} } @headers), "headers do not contain session cookie");
};

done_testing();
